#include <windows.h>

#include <fstream>
#include <iostream>
#include <iomanip>
#include <ostream>
#include <sstream>
#include <codecvt>
#include <locale>

#include "utils.hh"
#include "logger.hh"

// by default
static bool stdout_enabled=true;
static bool timestamp_enabled=false;
static std::wstring logfilename=L"tracer.log";

void Log_init()
{
    _wunlink (logfilename.c_str());
};

Log::~Log()
{
    os << std::endl;
    if (stdout_enabled)
        std::wcout << os.str();

    // https://stackoverflow.com/questions/6932409/writing-a-string-to-the-end-of-a-file-c/6932446
    std::wofstream out;

    // std::ios::app is the open mode "append" meaning
    // new data will be written to the end of the file.
    out.open(logfilename, std::ios::app);

    // https://stackoverflow.com/questions/9859020/windows-unicode-c-stream-output-failure/9869272#9869272
    const std::locale utf8_locale = std::locale(std::locale(), new std::codecvt_utf8<wchar_t>());
    out.imbue(utf8_locale);

    SYSTEMTIME t;
    GetLocalTime (&t);
    if (timestamp_enabled)
        out << wstrfmt(L"[%04d-%02d-%02d %02d:%02d:%02d:%03d] ", t.wYear, t.wMonth, t.wDay, t.wHour, t.wMinute, t.wSecond, t.wMilliseconds);
    out << os.str();
}

// http://stackoverflow.com/questions/11376288/fast-computing-of-log2-for-64-bit-integers
static const int tab64[64]=
{
    63,  0, 58,  1, 59, 47, 53,  2,
    60, 39, 48, 27, 54, 33, 42,  3,
    61, 51, 37, 40, 49, 18, 28, 20,
    55, 30, 34, 11, 43, 14, 22,  4,
    62, 57, 46, 52, 38, 26, 32, 41,
    50, 36, 17, 19, 29, 10, 13, 21,
    56, 45, 25, 31, 35, 16,  9, 12,
    44, 24, 15,  8, 23,  7,  6,  5
};

uint64_t uint64_log2 (uint64_t value)
{
    value |= value >> 1;
    value |= value >> 2;
    value |= value >> 4;
    value |= value >> 8;
    value |= value >> 16;
    value |= value >> 32;
    return tab64[((uint64_t)((value - (value >> 1))*0x07EDD5E59A4E28C2)) >> 58];
}

unsigned division_ceil (unsigned x, unsigned y)
{
    // https://stackoverflow.com/questions/2745074/fast-ceiling-of-an-integer-division-in-c-c
    return (x + y - 1) / y;
};

void Log::hexdump(BYTE *buf, size_t size, size_t ofs)
{
    size_t last_address_to_be_printed=ofs+size;
    unsigned fill_zeroes;
    if (last_address_to_be_printed==0)
    {
        fill_zeroes=0;
    }
    else
    {
        unsigned binary_digits=uint64_log2(last_address_to_be_printed)+1;
        // say, 11/4=2, but to print 11 binary digits, we want 3 hexadecimal digits
        // so the result of division must be "ceiled" to 3:
        fill_zeroes=division_ceil(binary_digits, 4);
    };

    std::wofstream out;
    // std::ios::app is the open mode "append" meaning
    // new data will be written to the end of the file.
    out.open(logfilename, std::ios::app);

    // https://stackoverflow.com/questions/9859020/windows-unicode-c-stream-output-failure/9869272#9869272
    const std::locale utf8_locale = std::locale(std::locale(), new std::codecvt_utf8<wchar_t>());
    out.imbue(utf8_locale);

    SYSTEMTIME t;
    GetLocalTime (&t);

    size_t pos=0;
    unsigned starting_offset=0;
    unsigned i;

    out << std::setfill(L'0');
    std::wcout << std::setfill(L'0');

    while (size-pos)
    {
        size_t wpn;
        if ((size-pos)>16)
            wpn=16;
        else
            wpn=size-pos;

        if (timestamp_enabled)
            out << wstrfmt(L"[%04d-%02d-%02d %02d:%02d:%02d:%03d] ", t.wYear, t.wMonth, t.wDay, t.wHour, t.wMinute, t.wSecond, t.wMilliseconds);
        out << "0x" << std::hex << std::setw(fill_zeroes) << (starting_offset + pos + ofs) << " ";
        std::wcout << "0x" << std::hex << std::setw(fill_zeroes) << (starting_offset + pos + ofs) << " ";

        for (i=0; i<wpn; i++)
        {
            out << std::setw(2) << (uint8_t)buf[pos+i] << ((i==7) ? '-' : ' ');
            std::wcout << std::setw(2) << (uint8_t)buf[pos+i] << ((i==7) ? '-' : ' ');
        };

        if (wpn<16)
            for (i=0; i<16-wpn; i++)
            {
                out << "   ";
                std::wcout << "   ";
            };

        out << "\"";
        std::wcout << "\"";

        for (i=0; i<wpn; i++)
        {
            out << (isprint (buf[pos+i]) ? (char)buf[pos+i] : '.');
            std::wcout << (isprint (buf[pos+i]) ? (char)buf[pos+i] : '.');
        };

        if (wpn<16)
            for (i=0; i<16-wpn; i++)
            {
                out << " ";
                std::wcout << " ";
            };

        out << "\"" << std::endl;
        std::wcout << "\"" << std::endl;

        pos+=wpn;
    };

}

#if 1
int main()
{
    const int count = 3;
    Log().Get() << L"A loop with "    << count << L" iterations";
    for (int i = 0; i != count; ++i)
    {
        Log().Get()        << L"the counter i = " << i;
    }
    BYTE buf[0x2000];
    for (int i=0; i<0x2000; i++)
        buf[i]=i&0xff;
    Log().hexdump(buf, 0x1100, 0);
};
#endif

// test
//LOGGER << utf8_to_utf16 ("Денис Юричев [yʊritʃev]");

/* vim: set expandtab ts=4 sw=4 : */
