#include <algorithm>
#include <iostream>
#include <list>
#include <string>
#include <locale>
#include <codecvt>

#include <assert.h>
#include <stdarg.h>
#include <stdint.h>

std::string strfmt (const char * szFormat, ...)
{
    va_list va;
    std::string rt;
    char * buf;
    int sz, bufsize;

    va_start (va, szFormat);

    sz=_vscprintf (szFormat, va);
    bufsize=sz+1;

    buf=(char*)malloc(bufsize);

    assert (buf);

    if (vsnprintf_s (buf, bufsize, sz, szFormat, va)==-1)
    {
        printf ("strfmt (%s...)\n", szFormat);
        printf ("_vsnprintf returned -1\n");
        assert (0);
    }
    else
        rt=buf;

    free (buf);

    return rt;
};

std::wstring wstrfmt (const wchar_t * szFormat, ...)
{
    va_list va;
    std::wstring rt;
    wchar_t * buf;
    int sz, bufsize;

    va_start (va, szFormat);

    sz=_vscwprintf (szFormat, va);
    bufsize=(sz+1)*2;

    buf=(wchar_t*)malloc(bufsize);

    assert (buf);

    if (_vsnwprintf_s (buf, bufsize, sz, szFormat, va)==-1)
    {
        wprintf (L"strfmt (%s...)\n", szFormat);
        wprintf (L"_vsnwprintf returned -1\n");
        assert (0);
    }
    else
        rt=buf;

    free (buf);

    return rt;
};

bool str_startswith (std::wstring prefix, std::wstring s)
{
    return std::equal(prefix.begin(), prefix.end(), s.begin());
};

std::wstring n_spaces (unsigned n)
{
    std::wstring rt;
    for (unsigned i=0; i<n; i++)
        rt=rt+L" ";
    return rt;
};

bool string_is_in_list_of_strings (const std::list<std::wstring> & l, const std::wstring & s)
{
    for (auto const & k : l)
    {
        if (k==s)
            return true;
    }
    return false;
};
void die (const std::wstring & msg)
{
    std::wcerr << msg;
    std::exit(EXIT_FAILURE);
}

// https://stackoverflow.com/questions/32055357/visual-studio-c-2015-stdcodecvt-with-char16-t-or-char32-t
std::wstring utf8_to_utf16(std::string utf8_string)
{
    return std::wstring_convert<std::codecvt_utf8_utf16<wchar_t>, wchar_t>{}.from_bytes(utf8_string);
}

std::string utf16_to_utf8(std::wstring utf16_string)
{
    return std::wstring_convert<std::codecvt_utf8_utf16<wchar_t>, wchar_t>{}.to_bytes(utf16_string);
}

// https://stackoverflow.com/questions/1070497/c-convert-hex-string-to-signed-integer
// https://linux.die.net/man/3/strtoul
uint64_t hexstring_to_int (std::string s)
{
    char *p;
#ifdef _WIN64
    unsigned n=strtoull(s.c_str(), &p, 16);
#else
    unsigned n=strtoul(s.c_str(), &p, 16);
#endif
    if (*p != 0)
    {
        assert (!"not a number");
    }
    return n;
};

// Rationale: couldn't find it in MSVC
void* my_memdup (void *src, size_t size)
{
    void* rt=malloc(size);
    assert (rt);
    memcpy (rt, src, size);
    return rt;
};

// Rationale: couldn't find it in MSVC
void my_bzero (void *dst, size_t size)
{
    memset(dst, 0, size);
};

// https://stackoverflow.com/questions/11635/case-insensitive-string-comparison-in-c
bool iequals(const std::string& a, const std::string& b)
{
    return std::equal(a.begin(), a.end(),
            b.begin(), b.end(),
            [](char a, char b) {
                return tolower(a) == tolower(b);
            });
}

bool iequals(const std::wstring& a, const std::wstring& b)
{
    return std::equal(a.begin(), a.end(),
            b.begin(), b.end(),
            [](wchar_t a, wchar_t b) {
                return towlower(a) == towlower(b);
            });
}

/* vim: set expandtab ts=4 sw=4 : */
